//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
// SPDX-FileCopyrightText: Copyright (c) 2023 NVIDIA CORPORATION & AFFILIATES.
//
//===----------------------------------------------------------------------===//

// MIT License
//
// Modifications Copyright (C) 2025 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

// UNSUPPORTED: c++11
// UNSUPPORTED: msvc && c++14, msvc && c++17

// No CTAD in C++14 or earlier
// UNSUPPORTED: c++14

#include <cuda/std/cassert>
#include <cuda/std/mdspan>

#define CHECK_MDSPAN(m, d)              \
  static_assert(m.is_exhaustive(), ""); \
  assert(m.data_handle() == d.data());  \
  assert(m.rank() == 2);                \
  assert(m.rank_dynamic() == 2);        \
  assert(m.extent(0) == 64);            \
  assert(m.extent(1) == 128)

int main(int, char**)
{
#ifdef __MDSPAN_USE_CLASS_TEMPLATE_ARGUMENT_DEDUCTION
  // TEST(TestMdspanCTAD, extents_object)
  {
    cuda::std::array<int, 1> d{42};
    cuda::std::mdspan m{d.data(), cuda::std::extents{64, 128}};

    CHECK_MDSPAN(m, d);
  }

  // TEST(TestMdspanCTAD, extents_object_move)
  {
    cuda::std::array<int, 1> d{42};
    // NOTE(HIPRTC): Internal issue #83.
    #if !defined(_CCCL_COMPILER_HIPRTC)
    cuda::std::mdspan m{d.data(), std::move(cuda::std::extents{64, 128})};
    #else
    cuda::std::mdspan m{d.data(), cuda::std::move(cuda::std::extents{64, 128})};
    #endif

    CHECK_MDSPAN(m, d);
  }

  // TEST(TestMdspanCTAD, extents_std_array)
  {
    cuda::std::array<int, 1> d{42};
    cuda::std::mdspan m{d.data(), cuda::std::array{64, 128}};

    CHECK_MDSPAN(m, d);
  }

  // TEST(TestMdspanCTAD, cptr_extents_std_array)
  {
    cuda::std::array<int, 1> d{42};
    const int* const ptr = d.data();
    cuda::std::mdspan m{ptr, cuda::std::array{64, 128}};

    static_assert(cuda::std::is_same<typename decltype(m)::element_type, const int>::value, "");

    CHECK_MDSPAN(m, d);
  }

  // extents from std::span
  {
    cuda::std::array<int, 1> d{42};
    cuda::std::array<int, 2> sarr{64, 128};
    cuda::std::mdspan m{d.data(), cuda::std::span{sarr}};

    CHECK_MDSPAN(m, d);
  }
#endif

  return 0;
}
